<?php
session_start();
require_once '../config/db.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get date range from request
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

try {
    // Get products data with sales information
    $stmt = $pdo->prepare("
        SELECT p.*,
               COALESCE(SUM(si.quantity), 0) as total_sold,
               COALESCE(SUM(si.total_price), 0) as total_revenue,
               COALESCE(AVG(si.unit_price), p.price) as avg_selling_price
        FROM products p
        LEFT JOIN sale_items si ON p.id = si.product_id
        LEFT JOIN sales s ON si.sale_id = s.id AND DATE(s.created_at) BETWEEN ? AND ?
        GROUP BY p.id
        ORDER BY total_sold DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $products = $stmt->fetchAll();

    // Set headers for Excel download
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment;filename="products_report_' . date('Y-m-d') . '.xls"');
    header('Cache-Control: max-age=0');

    // Output Excel content
    echo "Products Report\n";
    echo "Period: " . date('M d, Y', strtotime($start_date)) . " to " . date('M d, Y', strtotime($end_date)) . "\n\n";

    // Headers
    echo implode("\t", [
        'Product Code',
        'Name',
        'Category',
        'Current Stock',
        'Units Sold',
        'Cost Price',
        'Selling Price',
        'Avg. Selling Price',
        'Total Revenue',
        'Profit'
    ]) . "\n";

    // Data rows
    foreach ($products as $product) {
        $profit = $product['total_revenue'] - ($product['cost_price'] * $product['total_sold']);
        echo implode("\t", [
            $product['product_code'],
            $product['name'],
            $product['category'],
            $product['stock_quantity'],
            $product['total_sold'],
            number_format($product['cost_price'], 2),
            number_format($product['price'], 2),
            number_format($product['avg_selling_price'], 2),
            number_format($product['total_revenue'], 2),
            number_format($profit, 2)
        ]) . "\n";
    }

} catch (PDOException $e) {
    die('Error generating report: ' . $e->getMessage());
} 